<?php


namespace App\Filament\Resources\PerfilEmpresarialResource\RelationManagers;

use Filament\Forms;
use Filament\Tables;
use Filament\Resources\RelationManagers\RelationManager;
use Illuminate\Database\Eloquent\Builder;
use Filament\Notifications\Notification;
use Closure;


class ContactoClaveRelationManager extends RelationManager
{
    protected static string $relationship = 'contactosClave'; // relación definida en el modelo PerfilEmpresarial

    protected static ?string $title = 'Contactos clave';

    public function form(Forms\Form $form): Forms\Form
    {
        return $form->schema([
            Forms\Components\TextInput::make('nombre')->required()->label('Nombre del contacto'),
            Forms\Components\TextInput::make('apellidos')->required(),
            Forms\Components\TextInput::make('puesto_departamento')->required(),
            Forms\Components\TextInput::make('email')->required()->email(),
            Forms\Components\TextInput::make('lada_telefono')->required()->maxLength(3),
            Forms\Components\TextInput::make('telefono')->required()->maxLength(10),
            Forms\Components\FileUpload::make('logo_path')
                    ->label('Logo de la Empresa')
                    ->image()
                    ->avatar()
                    ->imageEditor()
                    ->circleCropper()
                    ->maxfiles(1)
                    ->maxSize(1024),// Tamaño máximo en KB
                    
            Forms\Components\TextInput::make('pagina_web')->url()->nullable(),

        ]);
    }

    public function table(Tables\Table $table): Tables\Table
    {
        return $table
            ->columns([
                Tables\Columns\TextColumn::make('nombre')->sortable(),
                Tables\Columns\TextColumn::make('apellidos')->sortable(),
                Tables\Columns\TextColumn::make('puesto_departamento'),
                Tables\Columns\TextColumn::make('email'),
                Tables\Columns\ImageColumn::make('logo_path'),
                
            ])
            ->headerActions([
                Tables\Actions\CreateAction::make()
                    ->mutateFormDataUsing(function (array $data): array {
                        $data['perfil_empresarial_id'] = $this->ownerRecord->id;
                        return $data;
                    })
                    ->visible(fn () => $this->ownerRecord->contactosClave()->count() < 3)
                    ->after(function () {
                        if ($this->ownerRecord->contactosClave()->count() >= 3) {
                            Notification::make() //Notificación de Filament para mostrar un mensaje al usuario
                                ->title('Límite alcanzado')
                                ->body('Ya has agregado el máximo de 3 contactos clave.')
                                ->danger()
                                ->send();
                        }
                    }),
            ])
            ->actions([
                Tables\Actions\EditAction::make(),
                Tables\Actions\DeleteAction::make(),
            ]);
    }
}
