<?php

namespace App\Filament\Resources;
use BezhanSalleh\FilamentShield\Traits\HasShieldPermissionsForm;
use App\Filament\Resources\UserResource\Pages;
use App\Filament\Resources\UserResource\RelationManagers;
use App\Models\User;
use Filament\Forms;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Select;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Filament\Tables\Filters\TernaryFilter;
use Filament\Tables\Columns\IconColumn;
use Filament\Tables\Filters\Filter;
use Illuminate\Support\Facades\Date;
use Spatie\Permission\Models\Permission;
use Filament\Forms\Components\CheckboxList;
use Filament\Forms\Components\MultiSelect;

class UserResource extends Resource
{
    

    protected static ?string $model = User::class;

    protected static ?string $navigationIcon = 'heroicon-o-users';

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                TextInput::make('name')
                    ->label('Nombre')
                    ->required()
                    ->maxLength(255),
                TextInput::make('email')
                    ->required()
                    ->email()
                    ->maxLength(255),
                Select::make('roles')
                ->multiple()
                ->preload()
                ->relationship('roles', 'name'),
                Select::make('tipo_membresia_id')
                    ->preload()
                    ->relationship(name: 'membresia', titleAttribute: 'nombre'),
                MultiSelect::make('permissions')
                    ->label('Permisos especiales')
                    // 1) Le indicamos la relación many-to-many y el campo “title”
                    ->relationship('permissions', 'name')
                    // 2) Personalizamos la etiqueta que ve el usuario:
                    //    “crear_foo (FooCategory)”, etc.
                    ->getOptionLabelUsing(fn (Permission $permission): string => 
                        "{$permission->name} ({$permission->category})"
                    )
                    ->searchable()    // buscador interno
                    ->preload()       // carga todas las opciones al abrir
                    ->multiple()      // selección múltiple
                    ->columnSpanFull()// ocupa todo el ancho del formulario
                    ->placeholder('Selecciona permisos específicos'),
                    TextInput::make('password')
                        ->password()
                        ->revealable()
                        ->visibleOn('create')
                        ->required()
                        ->minLength(8),
                            ]);
                    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                TextColumn::make('name')
                    ->label('Nombre')
                    ->sortable()
                    ->searchable(),
                TextColumn::make('email')
                    ->sortable()
                    ->searchable(),
               IconColumn::make('email_verified_at')
                    ->label('Verificado')
                    ->boolean()
                    ->getStateUsing(fn (User $record): bool => ! is_null($record->email_verified_at))
                    ->trueIcon('heroicon-o-check-circle')
                    ->falseIcon('heroicon-o-x-circle')
                    ->trueColor('success')
                    ->falseColor('danger'),
                TextColumn::make('roles.name'),
                TextColumn::make('membresia.nombre')
                    ->label('Membresía')
                    ->sortable()
                    ->searchable(),
            ])
            ->filters([
                TernaryFilter::make('email_verified_at')
                    ->label('Correo Verificado')
                    ->nullable(),
                Tables\Filters\Filter::make('verified')
                  ->query(fn (Builder $query): Builder => $query->whereNotNull('email_verified_at')),
                
            ])
            ->actions([
                Tables\Actions\EditAction::make(),
                Tables\Actions\Action::make('Verify')
                    ->icon('heroicon-o-check')
                    ->action(function (User $user) {
                        $user->email_verified_at = Date('Y-m-d H:i:s');
                        $user->save();
                    }),
                Tables\Actions\Action::make('Unverify')
                    ->icon('heroicon-o-x-circle')
                    ->action(function (User $user) {
                        $user->email_verified_at = null;
                        $user->save();
                    }),
                
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListUsers::route('/'),
            'create' => Pages\CreateUser::route('/create'),
            'edit' => Pages\EditUser::route('/{record}/edit'),
        ];
    }
    public static function getModelLabel(): string
    {
        return 'usuario';
    }
    public static function getModelPluralLabel(): string
    {
        return 'usuarios';
    }
}
