<?php

namespace App\Http\Controllers;

use App\Models\Post;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;

class UserActionController extends Controller
{
    

    public function misPublicaciones()
    {
        $posts = Post::where('user_id', Auth::id())
                    ->orderBy('created_at', 'desc')
                    ->paginate(2);

        return view('user-actions.publicaciones', compact('posts'));
    }
    public function crearPublicacion(Request $request)
    {
        // 1. Validar
        $data = $request->validate([
            'title'   => 'required|string|max:255',
            'content' => 'required|string',
            'images_path.*' => 'nullable|image|max:2048',
            'temas'   => 'required|array',
            'temas.*' => 'exists:temas,id',
        ]);

        // 2. Guardar imágenes (si hay)
        $imagesPath = [];
        if ($request->hasFile('images_path')) {
            foreach ($request->file('images_path') as $image) {
                $path = $image->store('posts', 'public'); // en storage/app/public/posts
                $imagesPath[] = $path;
            }
        }

        // 3. Crear el post
        $post = Post::create([
            'title'        => $data['title'],
            'content'      => $data['content'],
            'user_id'      => Auth::id(),
            'images_path'  => $imagesPath,
            'published_at' => now(),
        ]);

        // 4. Relacionar temas
        $post->temas()->attach($data['temas']);

        return redirect()->back()->with('success', 'Publicación creada correctamente.');
    }

    public function eliminarPublicacion(Post $post)
    {
        // Verificar que el usuario autenticado sea el autor
        if ($post->user_id !== Auth::id()) {
            return redirect()->back()->with('error', 'No tienes permiso para eliminar esta publicación.');
        }

        // Eliminar imágenes del storage si existen
        if (is_array($post->images_path)) {
            foreach ($post->images_path as $image) {
                if (Storage::disk('public')->exists($image)) {
                    Storage::disk('public')->delete($image);
                }
            }
        }

        // Eliminar la relación con temas
        $post->temas()->detach();

        // Eliminar la publicación
        $post->delete();

        return redirect()->back()->with('success', 'Publicación eliminada correctamente.');
    }
    public function editarPublicacion(Post $post)
    {
        // Verificar que el usuario autenticado sea el autor
        if ($post->user_id !== Auth::id()) {
            return redirect()->back()->with('error', 'No tienes permiso para editar esta publicación.');
        }

        // Obtener todos los temas para el select
        $temas = \App\Models\Tema::all();

        return view('user-actions.editar-publicacion', compact('post', 'temas'));
    }

    public function actualizarPublicacion(Request $request, Post $post)
    {
        // Verificar que el usuario autenticado sea el autor
        if ($post->user_id !== Auth::id()) {
            return redirect()->back()->with('error', 'No tienes permiso para actualizar esta publicación.');
        }

        // Validar los datos
        $data = $request->validate([
            'title'   => 'required|string|max:255',
            'content' => 'required|string',
            'images_path.*' => 'nullable|image|max:2048',
            'temas'   => 'required|array',
            'temas.*' => 'exists:temas,id',
        ]);

        // Guardar nuevas imágenes si hay
        $imagesPath = $post->images_path ?? [];
        if ($request->hasFile('images_path')) {
            foreach ($request->file('images_path') as $image) {
                $path = $image->store('posts', 'public');
                $imagesPath[] = $path;
            }
        }

        // Actualizar el post
        $post->update([
            'title'       => $data['title'],
            'content'     => $data['content'],
            'images_path' => $imagesPath,
        ]);

        // Actualizar la relación de temas
        $post->temas()->sync($data['temas']);

        return redirect()->back()->with('success', 'Publicación actualizada correctamente.');
    }

}
